<?php
/**
 * Plugin Name: BookingFlow
 * Plugin URI: https://bookingflow.nl
 * Description: Integreer BookingFlow reserveringssysteem op je WordPress website. Laat bezoekers direct boeken via een professionele booking widget.
 * Version: 1.0.0
 * Author: BookingFlow
 * Author URI: https://bookingflow.nl
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: bookingflow
 * Requires at least: 5.8
 * Requires PHP: 7.4
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('BOOKINGFLOW_VERSION', '1.0.0');
define('BOOKINGFLOW_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('BOOKINGFLOW_PLUGIN_URL', plugin_dir_url(__FILE__));
define('BOOKINGFLOW_BASE_URL', 'https://bookingflow.nl');

/**
 * Main BookingFlow Plugin Class
 */
class BookingFlow_Plugin {

    /**
     * Constructor
     */
    public function __construct() {
        // Register activation hook
        register_activation_hook(__FILE__, array($this, 'activate'));

        // Initialize plugin
        add_action('plugins_loaded', array($this, 'init'));

        // Admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));

        // Admin scripts
        add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));

        // Register shortcode
        add_shortcode('bookingflow', array($this, 'shortcode'));

        // Register Gutenberg block
        add_action('init', array($this, 'register_block'));

        // Add settings link on plugin page
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'add_settings_link'));
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Set default options
        if (!get_option('bookingflow_tenant_slug')) {
            add_option('bookingflow_tenant_slug', '');
        }
        if (!get_option('bookingflow_display_mode')) {
            add_option('bookingflow_display_mode', 'embed');
        }
        if (!get_option('bookingflow_width')) {
            add_option('bookingflow_width', '100%');
        }
        if (!get_option('bookingflow_height')) {
            add_option('bookingflow_height', '800px');
        }
    }

    /**
     * Initialize plugin
     */
    public function init() {
        // Load text domain for translations
        load_plugin_textdomain('bookingflow', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            __('BookingFlow Instellingen', 'bookingflow'),
            __('BookingFlow', 'bookingflow'),
            'manage_options',
            'bookingflow',
            array($this, 'settings_page'),
            'dashicons-calendar-alt',
            30
        );
    }

    /**
     * Admin scripts
     */
    public function admin_scripts($hook) {
        if ($hook !== 'toplevel_page_bookingflow') {
            return;
        }

        wp_enqueue_style('bookingflow-admin', BOOKINGFLOW_PLUGIN_URL . 'assets/admin.css', array(), BOOKINGFLOW_VERSION);
    }

    /**
     * Settings page
     */
    public function settings_page() {
        // Save settings
        if (isset($_POST['bookingflow_save_settings']) && check_admin_referer('bookingflow_settings')) {
            update_option('bookingflow_tenant_slug', sanitize_text_field($_POST['tenant_slug']));
            update_option('bookingflow_display_mode', sanitize_text_field($_POST['display_mode']));
            update_option('bookingflow_width', sanitize_text_field($_POST['width']));
            update_option('bookingflow_height', sanitize_text_field($_POST['height']));

            echo '<div class="notice notice-success"><p>' . __('Instellingen opgeslagen!', 'bookingflow') . '</p></div>';
        }

        $tenant_slug = get_option('bookingflow_tenant_slug', '');
        $display_mode = get_option('bookingflow_display_mode', 'embed');
        $width = get_option('bookingflow_width', '100%');
        $height = get_option('bookingflow_height', '800px');

        include BOOKINGFLOW_PLUGIN_DIR . 'templates/settings.php';
    }

    /**
     * Shortcode handler
     */
    public function shortcode($atts) {
        $atts = shortcode_atts(array(
            'tenant' => get_option('bookingflow_tenant_slug', ''),
            'width' => get_option('bookingflow_width', '100%'),
            'height' => get_option('bookingflow_height', '800px'),
            'mode' => get_option('bookingflow_display_mode', 'embed'),
        ), $atts, 'bookingflow');

        if (empty($atts['tenant'])) {
            return '<p style="color: red;">' . __('BookingFlow: Geen tenant slug ingesteld. Configureer de plugin in de instellingen.', 'bookingflow') . '</p>';
        }

        $booking_url = BOOKINGFLOW_BASE_URL . '/booking/' . esc_attr($atts['tenant']);

        if ($atts['mode'] === 'button') {
            return $this->render_button($booking_url);
        }

        return $this->render_embed($booking_url, $atts['width'], $atts['height']);
    }

    /**
     * Render embed mode
     */
    private function render_embed($url, $width, $height) {
        return sprintf(
            '<div class="bookingflow-embed-container" style="width: %s; max-width: 100%%;">
                <iframe src="%s"
                    width="100%%"
                    height="%s"
                    frameborder="0"
                    scrolling="auto"
                    style="border: none; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
                </iframe>
            </div>',
            esc_attr($width),
            esc_url($url),
            esc_attr($height)
        );
    }

    /**
     * Render button mode
     */
    private function render_button($url) {
        return sprintf(
            '<div class="bookingflow-button-container" style="text-align: center; margin: 20px 0;">
                <a href="%s"
                   class="bookingflow-button"
                   target="_blank"
                   style="display: inline-block; background: #2563EB; color: white; padding: 14px 32px; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 16px; transition: background 0.2s;">
                    %s
                </a>
            </div>
            <style>
                .bookingflow-button:hover {
                    background: #1E40AF !important;
                    color: white !important;
                }
            </style>',
            esc_url($url),
            __('Reserveer Nu', 'bookingflow')
        );
    }

    /**
     * Register Gutenberg block
     */
    public function register_block() {
        if (!function_exists('register_block_type')) {
            return;
        }

        wp_register_script(
            'bookingflow-block',
            BOOKINGFLOW_PLUGIN_URL . 'assets/block.js',
            array('wp-blocks', 'wp-element', 'wp-editor', 'wp-components'),
            BOOKINGFLOW_VERSION
        );

        register_block_type('bookingflow/booking-widget', array(
            'editor_script' => 'bookingflow-block',
            'render_callback' => array($this, 'shortcode'),
            'attributes' => array(
                'tenant' => array(
                    'type' => 'string',
                    'default' => get_option('bookingflow_tenant_slug', ''),
                ),
                'width' => array(
                    'type' => 'string',
                    'default' => get_option('bookingflow_width', '100%'),
                ),
                'height' => array(
                    'type' => 'string',
                    'default' => get_option('bookingflow_height', '800px'),
                ),
                'mode' => array(
                    'type' => 'string',
                    'default' => get_option('bookingflow_display_mode', 'embed'),
                ),
            ),
        ));
    }

    /**
     * Add settings link on plugins page
     */
    public function add_settings_link($links) {
        $settings_link = '<a href="admin.php?page=bookingflow">' . __('Instellingen', 'bookingflow') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
}

// Initialize plugin
new BookingFlow_Plugin();
